"use strict";

/**
 * This class displays the GUI and handle all events.
 */
var gui = (function() {

    var mainDiv;
    var mainInternalDiv;
    var messageBox;
    var bottomLine;
    var statusLine;
    var statusMessage;
    var editBox;
    var comboBox;
    var buttonList;
    var clipboard;
    var buttons = [];

    var minimalButtonWidth = -1;

    var guiProtocolService = null;    
    var statusMessageAnimation;
    var pluginTagObserver;
    var thiz;
    
    ///////////////////////////////////////////////////
    // Initialization
    ///////////////////////////////////////////////////

    // Initializes the GUI (Step 1 of initialization)
    function initialize(pluginDiv) {
        // Check for supported type
        var type = $(pluginDiv).data('type');
        if (!isExtensionTypeSupported(type)) {
            logDebug("Type '" + type + "' is not supported!");
            return;
        }

        logDebug("Initializing GUI...");
        
        mainDiv = pluginDiv;
        
        // Mark gui as started to avoid multiple initializations.
        $(mainDiv).addClass("travic-sign-started");
             
        // Hide all inner elements (e.g. alternative text for installation)
        hideAlternativeContent();
             
        $(window).on('beforeunload', function() {
            thiz.closeGui();
        });
             
        loadHtmlTemplate();
    }
             
    // Loads the HTML template (Step 2 of initialization)
    function loadHtmlTemplate() {
        logDebug("Loading HTML template (initial hidden)");

        // Create a hidden element containing the whole GUI
        mainInternalDiv = $("<div>", {id: "travic-sign-internal"});
        $(mainInternalDiv).hide();
        $(mainDiv).append(mainInternalDiv);
        
        var fragment = newHtmlFragment();
        fragment.load($(mainInternalDiv), loadHtmlTemplateSuccess, loadHtmlTemplateError);
    }
    
    // Called when loading HTML template failed
    function loadHtmlTemplateError(status) {
        logDebug("GUI initialization failed. HTML Fragment could not be loaded. Status: " + status);
    }
        
    // Initializes the GUI after the HTML template has been loaded and inserted 
    // into the current page. 
    function loadHtmlTemplateSuccess() {
        initializeDisplay();
        initialzeButtonHandlers();
        initializeKeyHandlers();

        if (!thiz.hasModeFlag("IgnoreGuiParameters")) {
           evaluateGuiParameters();
        }

        logDebug("GUI initialization finished.");
        
        startGuiProtocolService();
    }
       
    /**
     * Makes the inner HTML GUI tag visible
     */
    function showInternalGuiDiv() {
        logDebug("Making GUI visible");

        // Remove everything except the internal div    
        $(mainInternalDiv).show();
    }   
    
    /**
     * Hides the alternative content of the main div
     * This content should only be shown when any of the components are missing.
     */
    function showAlternativeContent() {
        logDebug("Showing alternative content");
        var mainInternalDivSelector = "#" + $(mainInternalDiv).attr("id");
        $(mainDiv).contents().not(mainInternalDivSelector).show();
    }
 
    /**
     * Hides the alternative content of the main div
     * This content should only be shown when any of the components are missing.
     */
    function hideAlternativeContent() {
        logDebug("Hiding alternative content");
        var mainInternalDivSelector = "#" + $(mainInternalDiv).attr("id");
        $(mainDiv).contents().not(mainInternalDivSelector).hide();
    }
           
    /**
     * Checks if alternative content is defined.
     */
    function hasAlternativeContent() {
        var mainInternalDivSelector = "#" + $(mainInternalDiv).attr("id");
        
        // Remove the internal div
        var clonedMainDiv = $(mainDiv).clone();
        $(clonedMainDiv).find(mainInternalDivSelector).remove();
             
        // Check html content
        var htmlContent = $(clonedMainDiv).html().trim();
        return (htmlContent.length > 0);
    }   
       
    // Initializes the GuiProtocolService (Step 3 of initialization)
    function startGuiProtocolService() {
        logDebug("Starting GUI protocol service");
        if (guiProtocolService !== null) {
            logDebug("GUI protocol service already started");
            return;
        }
        
        // Initialize protocol service and start plugin main
        guiProtocolService = newGuiProtocolService(thiz, 
                                                   $(mainDiv).data(), 
                                                   startGuiProtocolServiceSuccess, 
                                                   onGuiGuiProtocolServiceDisconnect,
                                                   onGuiGuiProtocolServiceFirstMessageReceived);
        guiProtocolService.connect();
    }
    
    // Called when start of GuiProtocolService succeeds.
    function startGuiProtocolServiceSuccess() {
        logDebug("GUI protocol service successfully started");

        guiProtocolService.requestNativeHostVersion();             
        
        if (isInstallationTestRun()) {
            logDebug("Installation test mode. Skipping further initilization");
        }
        else {
            logDebug("Starting PluginMain");
            guiProtocolService.startPluginMain();        
        }
    }
        
    // Called when GuiProtocolService is disconnecting
    function onGuiGuiProtocolServiceDisconnect() {
        logDebug("Disconnecting GUI protocol service");
        
        var event = new CustomEvent("travic-sign-missing-native-host", {"bubbles": true,});
        dispatchEventOnPageElement(event);                                 

        if (hasAlternativeContent()) {
            showAlternativeContent();
        }
        else {
            guiProtocolService.sendErrorMessage(207, "Native messaging host disconnected / not found");
        }
        
        thiz.closeGui();        
    }        
       
    // Called when GuiProtocolService has received the first message from background script
    function onGuiGuiProtocolServiceFirstMessageReceived() {
        showInternalGuiDiv();
    }          
       
    // Initialize main display properties
    function initializeDisplay() {
        logDebug("Initializing display...");

        // Remember all elements
        messageBox    = $("#travic-sign-message");
        bottomLine    = $("#travic-sign-bottom-line");
        statusLine    = $("#travic-sign-status-line");
        statusMessage = $("#travic-sign-status-message");
        editBox       = $("#travic-sign-editbox");
        comboBox      = $("#travic-sign-combobox");
        buttonList    = $("#travic-sign-button-list");
        buttons[1]    = $("#travic-sign-button1");
        buttons[2]    = $("#travic-sign-button2");
        buttons[3]    = $("#travic-sign-button3");
        
        // This element is deprecated. The copy operation is done by the native backend since V3.1.
        // So this element can be removed in the near future.
        clipboard     = $("#travic-sign-clipboard");

        // Set initial height
        var height = thiz.getParameter("height");
        if (height) {
            thiz.setHeight(height);
        }

        // Set initial width
        var width = thiz.getParameter("width");
        if (width) {
            thiz.setWidth(width);
        }

        // Hide the clipboard element if not done by CSS (happens sporadic on Firefox)
        $(clipboard).hide();
        
        thiz.resize();
        registerResizeEventHandler();

        logDebug("Initialization of display finished");
    }
    
    // Registers event handlers for all buttons
    function initialzeButtonHandlers() {
        logDebug("Initializing button handlers...");

        $(buttons[1]).click(function() {
            thiz.handleButtonClicked(1);
            return false;
        });
        $(buttons[2]).click(function() {
            thiz.handleButtonClicked(2);
            return false;
        });
        $(buttons[3]).click(function() {
            thiz.handleButtonClicked(3);
            return false;
        });
        
        logDebug("Initialization of button handlers finished");
    }

    // Registers event handlers for editbox and combobox
    function initializeKeyHandlers() {
        logDebug("Initializing key handlers...");
        
        // Handle keys in editBox
        $(editBox).keypress(function(e) {
            if (e.which == 10 || e.which == 13) {
                e.stopImmediatePropagation();
                e.preventDefault();
                thiz.handleDefaultButtonClicked();
            }
        });

        // Handle keys in comboBox
        $(comboBox).keypress(function(e) {
            if (e.which == 10 || e.which == 13) {
                e.stopImmediatePropagation();
                e.preventDefault();
                thiz.handleDefaultButtonClicked();
            }
        });

        logDebug("Initialization of key handlers finished");
    }

    // Evaluates simple GUI parameters like Font or textcolour
    function evaluateGuiParameters() {
        logDebug("Evaluating GUI parameters...");

        // Set font size
        var fontSize = thiz.getParameter("FontSize");
        if (fontSize) {
            thiz.setFontSize(fontSize);
        }

        // Set font
        var fontName = thiz.getParameter("GuiFontName");
        if (fontName) {
            thiz.setFontName(fontName);
        }

        // Set background color (1)
        var backgroundColor1 = thiz.getParameter("BackgroundColour1");
        if (backgroundColor1) {
            thiz.setBackgroundColor(backgroundColor1);
        }

        // Set text color (1)
        var textColor1 = thiz.getParameter("TextColour1");
        if (textColor1) {
            thiz.setTextColor(textColor1);
        }

        // Set background color (2)
        var backgroundColor2 = thiz.getParameter("BackgroundColour2");
        if (backgroundColor2) {
            thiz.setEditBoxBackgroundColor(backgroundColor2);
            thiz.setComboBoxBackgroundColor(backgroundColor2);
        }

        // Set text color (2)
        var textColor2 = thiz.getParameter("TextColour2");
        if (textColor2) {
            thiz.setEditBoxTextColor(textColor2);
            thiz.setComboBoxTextColor(textColor2);
        }

        logDebug("Evaluation of GUI parameters finished");
    }   

    // Registers event handlers for resizing
    function registerResizeEventHandler() {
        var observerConfig = {
            childList : false,
            attributes : true,
            characterData : true,
            subtree : true
        };

        var observerElement = $(mainDiv).get(0);

        var observer = new MutationObserver(function() {
            // Pause observer to avoid recursion
            observer.disconnect();

            thiz.resize();

            // Restart observer
            observer.observe(observerElement, observerConfig);
        });

        observer.observe(observerElement, observerConfig);
    }        

    // Calculates the maximum height for the message box
    function adjustMessageBoxHeight() {
        var totalHeight = $(mainDiv).height();

        var messageBoxPadding = $(messageBox).outerHeight(true) - $(messageBox).height();
        var bottomLineHeight = bottomLine.outerHeight(true);

        var newMessageBoxHeight = totalHeight - bottomLineHeight - messageBoxPadding;
        $(messageBox).height(newMessageBoxHeight);
    }
        
    // Calculates the width of each element in the bottom line
    function adjustBottomLineSize() {
        var totalWidth = $(bottomLine).width();
        var buttonListWidth = $(buttonList).outerWidth(true);

        // Calculate width for status line (Notice: bottomLine has box-sizing: border-box)
        // 5px added because of rendering problem with Chrome 49 on Mac OS X
        var newStatusLineWidth = totalWidth - buttonListWidth - 5;
        $(statusLine).outerWidth(newStatusLineWidth);

        // Use the same height for all elements
        var buttonHeight = $(buttonList).height();
        if (buttonHeight > 0) {
            $(statusLine).height(buttonHeight);
        }
    }
        
    function adjustButtonWidth() {
        // Reset button with
        for (var buttonNo in buttons) {
            $(buttons[buttonNo]).width("auto");
        }

        // Calcalute max. button width
        var button1Width = thiz.isButtonVisible(1) ? $(buttons[1]).outerWidth() : 0;
        var button2Width = thiz.isButtonVisible(2) ? $(buttons[2]).outerWidth() : 0;
        var button3Width = thiz.isButtonVisible(3) ? $(buttons[3]).outerWidth() : 0;
        var maxWidth = Math.max(button1Width, button2Width, button3Width);

        // Set new width for all buttons
        for (buttonNo in buttons) {
            $(buttons[buttonNo]).width(maxWidth);
        }
    }
    
    ///////////////////////////////////////////////////////////////
    // Miscellaneous
    ///////////////////////////////////////////////////////////////
    
    function logDebug(message) {
        if (optionsService.isDebugEnabled()) {
            var dateString = formatDateString(new Date());
            console.debug(dateString + " - " + PRODUCT_NAME + " [Gui]: " + message);
        }
    }

    function logError(message) {
        var dateString = formatDateString(new Date());
        console.error(dateString + " - " + PRODUCT_NAME + " [Gui]: " + message);
    }

    /**
     * Start an observer to wait for new plugins added to the DOM.
     */         
    function startPluginObserver() {
        logDebug("Starting observer to wait for plugin instances to appear");
        
        // Configuration of the observer:
        var observerConfig = {
            childList : true,
            attributes : false,
            characterData : false,
            subtree : true
        };

        // Create an observer instance
        pluginTagObserver = new MutationObserver(function(mutations) {
            thiz.startGuiIfAvailable();
        });

        // Pass in the target node, as well as the observer options
        pluginTagObserver.observe(document, observerConfig);
        
        logDebug("Observer started");
    }

    /**
     * Stops the plugin observer.
     */         
    function stopPluginObserver() {
        if (pluginTagObserver) {
            logDebug("Stopping observer for plugin instances");
            
            pluginTagObserver.disconnect();
            pluginTagObserver = undefined;
        }
    }
    
    /**
     * Searches recursively for an Iframe with name <code>iframeName</code> 
     * starting at the given document.
     */        
    function findIframe(iframeName, document) {
        logDebug("Searching for iframe with name '" + iframeName + " in URL: " + document.URL);        
                   
        var iFrame = $("iframe[name='" + iframeName + "']", document);        
        if (iFrame.length) {
            logDebug("iFrame found: " + iFrame[0].src);                               
            return $(iFrame[0]);
        }
        
        var result = null;
        $("iframe", document).each(function() {
            logDebug("  - Inner iFrame found. Name: '" + this.name + "', URL: " + this.src);        
            iFrame = findIframe(iframeName, $(this).contents()[0]);
            if (iFrame) {
                logDebug("iFrame found: " + iFrame.src);        
                
                result = $(iFrame);
                return false;
            }
        });
        
        return result;
    }
     
    // Checks for installation test modus
    function isInstallationTestRun() {
        var installationTestFlag = thiz.getParameter("InstallationTest");
        return (installationTestFlag === true);
    }

    // Dispatch event on web page element
    function dispatchEventOnPageElement(event) {
    	logDebug("Dispatching event '" + event.type + "'");
        var targetElement = null;
        
        var targetElementId = thiz.getParameter("EventElementId");
        if (targetElementId && (targetElementId.length > 0)) {
            targetElement = $("#" + targetElementId, $(mainDiv)[0].ownerDocument);
        }

        // Use plugin div as default
        if (!targetElement || !targetElement.length) {
            targetElement = $(mainDiv);
        }
        if (targetElement && targetElement[0]) {
            targetElement[0].dispatchEvent(event);
        }
    }
    
    ///////////////////////////////////////////////////////////////
    // Public interface
    ///////////////////////////////////////////////////////////////
        
    thiz = {

        closeGui : function() {
            logDebug("Shutting down...");
            
            thiz.stopAllAnimations();
            
            if (guiProtocolService) {
                guiProtocolService.disconnect();
                guiProtocolService = null;
            }
            
            var event = new CustomEvent("travic-sign-closed", {"bubbles": true});
            dispatchEventOnPageElement(event);                                 

            $(mainInternalDiv).remove();
            $(mainDiv).addClass("travic-sign-closed");
            mainDiv = undefined;
    
            logDebug("Shutdown complete");
        },

        ///////////////////////////////////////////////////
        // Display functions
        ///////////////////////////////////////////////////

        // Recalculate size of some elements
        resize : function() {
            // All buttons have minimum width, but not the same width.
            // If otherwise wanted, uncomment the following line.
            // adjustButtonWidth();
            adjustBottomLineSize();
            adjustMessageBoxHeight();
        },


        calculateAndSetMinimalButtonWidth : function(standardButtonWidthText) {
            var savedButtonText = thiz.getButtonText(1);

            // Calculate minimal button width
            thiz.setButtonText(1, standardButtonWidthText);
            minimalButtonWidth = $(buttons[1]).outerWidth();

            thiz.setButtonText(1, savedButtonText);

            // Set minimal button width
            if (minimalButtonWidth >= 0) {
                for (var buttonNo in buttons) {
                    $(buttons[buttonNo]).css("min-width", minimalButtonWidth);
                }
            }

            thiz.resize();
        },
        
        setHeight : function(height) {
            $(mainDiv).height(height);
        },

        getHeight : function() {
            return $(mainDiv).height();
        },

        setWidth : function(width) {
            $(mainDiv).width(width);
        },

        getWidth : function() {
            return $(mainDiv).width();
        },

        setFontSize : function(fontSize) {
            $(mainDiv).css("font-size", fontSize + "px");
        },

        setFontName : function(font) {
            $(mainDiv).css("font-family", font);
        },

        getFontSize : function() {
            return $(mainDiv).css("font-size");
        },

        setBackgroundColor : function(color) {
            $(mainDiv).css("background-color", color);
        },

        setTextColor : function(color) {
            $(mainDiv).css("color", color);
        },

        setMessage : function(message) {
            $(messageBox).text(message);
            $(messageBox).scrollTop(0);
            $(messageBox).removeClass("error");
        },

        setErrorMessage : function(message) {
            thiz.setMessage(message);
            $(messageBox).addClass("error");
        },

        getMessage : function() {
            return $(messageBox).text();
        },

        setMessageFontFamily : function(fontFamily) {
            $(messageBox).removeClass("monospace");
            $(messageBox).removeClass("sans-serif");

            if (fontFamily === "monospace") {
                $(messageBox).addClass("monospace");
            }
            else {
                $(messageBox).addClass("sans-serif");
            }
        },

        setStatusMessage : function(message) {
            $(statusMessage).text(message);
        },

        getStatusMessage : function() {
            return $(statusMessage).text();
        },

        startStatusMessageAnimation : function(message, character, interval) {
            if (statusMessageAnimation) {
                statusMessageAnimation.stop();
            }
            
            statusMessageAnimation = newMessageAnimation(statusMessage, message, character, interval);
            statusMessageAnimation.start();
        },

        stopStatusMessageAnimation : function() {
            if (statusMessageAnimation) {
                statusMessageAnimation.stop();
                statusMessageAnimation = null;
            }
        },

        getEditBoxText : function() {
            return $(editBox).val();
        },

        setEditBoxText : function(text) {
            $(editBox).val(text);
        },

        setEditBoxPlaceholder : function(text) {
            $(editBox).attr("placeholder", text);
        },

        setEditBoxName : function(newName) {
            // Remove old name
            var oldName = $(editBox).data("name");
            if (oldName) {
                $(editBox).removeClass("travic-sign-editbox-" + oldName);
            }
            $(editBox).data("name", "");

            // Set new name
            if (newName) {
                newName = newName.toLowerCase().replace(/\s/gi, "");
                $(editBox).data("name", newName);

                if (newName.length > 0) {
                    $(editBox).addClass("travic-sign-editbox-" + newName);
                }
            }
        },


        setEditBoxSize : function(size) {
            $(editBox).attr("size", size);
        },

        clearEditBoxSize : function(size) {
            $(editBox).removeAttr("size");
        },

        setEditBoxTypeClear : function() {
            $(editBox).attr("type", "text");
        },

        setEditBoxTypePassword : function() {
            $(editBox).attr("type", "password");
        },

        hideEditBox : function() {
            $(editBox).hide();
        },

        showEditBox : function() {
            $(editBox).show();
        },

        focusEditBox : function() {
            // This is a workaround for Firefox.
            // Without setTimeout Firefox doesn't focus the edit box.
            setTimeout(function() {
                $(editBox).focus();
            },0);
        },

        isEditBoxVisible : function() {
            return ($(editBox).is(":visible"));
        },

        disableEditBox : function() {
            $(editBox).prop("disabled", true);
        },

        enableEditBox : function() {
            $(editBox).prop("disabled", false);
        },

        setEditBoxBackgroundColor : function(color) {
            $(editBox).css("background-color", color);
        },

        setEditBoxTextColor : function(color) {
            $(editBox).css("color", color);
        },

        hideComboBox : function() {
            $(comboBox).hide();
        },

        showComboBox : function() {
            $(comboBox).show();
        },

        focusComboBox : function() {
            // This is a workaround for Firefox.
            // Without setTimeout Firefox doesn't focus the combo box.
            setTimeout(function() {
                $(comboBox).focus();
            },0);
        },

        isComboBoxVisible : function() {
            return ($(comboBox).is(":visible"));
        },

        disableComboBox : function() {
            $(comboBox).prop("disabled", true);
        },

        enableComboBox : function() {
            $(comboBox).prop("disabled", false);
        },

        setComboBoxBackgroundColor : function(color) {
            $(comboBox).css("background-color", color);
        },

        setComboBoxTextColor : function(color) {
            $(comboBox).css("color", color);
        },

        getSelectedComboboxIndex : function() {
            return $(comboBox).prop('selectedIndex');
        },

        getSelectedComboboxText : function() {
            return $(comboBox).val();
        },

        clearComboBox : function() {
            $(comboBox).empty();
        },

        addComboBoxEntry : function(text, value) {
            $(comboBox).append(new Option(text, value));
        },

        setButtonText : function(buttonNo, text) {
            $(buttons[buttonNo]).text(text);
        },

        getButtonText : function(buttonNo) {
            return $(buttons[buttonNo]).prop('value');
        },

        setButtonName : function(buttonNo, newName) {
            // Remove old name
            var oldName = $(buttons[buttonNo]).data("name");
            if (oldName) {
                $(buttons[buttonNo]).removeClass("travic-sign-button-" + oldName);
            }
            $(buttons[buttonNo]).data("name", "");

            // Set new name
            if (newName) {
                newName = newName.toLowerCase().replace(/\s/gi, "");
                $(buttons[buttonNo]).data("name", newName);

                if (newName.length > 0) {
                    $(buttons[buttonNo]).addClass("travic-sign-button-" + newName);
                }
            }
        },

        enableButton : function(buttonNo) {
            $(buttons[buttonNo]).prop("disabled", false);
        },

        disableButton : function(buttonNo) {
            $(buttons[buttonNo]).prop("disabled", true);
        },

        showButton : function(buttonNo) {
            $(buttons[buttonNo]).show();
        },

        hideButton : function(buttonNo) {
            $(buttons[buttonNo]).hide();
        },

        isButtonVisible : function(buttonNo) {
            return $(buttons[buttonNo]).is(':visible');
        },

        focusButton : function(buttonNo) {
            $(buttons[buttonNo]).focus();
        },

        resetDefaultButton : function() {
            for (var buttonNo in buttons) {
                $(buttons[buttonNo]).removeClass("travic-sign-default-button");
            }
        },

        setDefaultButton : function(buttonNo) {
            thiz.resetDefaultButton();
            $(buttons[buttonNo]).addClass("travic-sign-default-button");
        },

        getDefaultButton : function() {
            for (var buttonNo in buttons) {
                if ($(buttons[buttonNo]).hasClass("travic-sign-default-button")) {
                    return buttonNo;
                }
            }
            return -1;
        },

        disableAll : function() {
            thiz.disableAllButtons();
            thiz.disableEditBox();
            thiz.disableComboBox();
        },

        disableAllButtons : function() {
            for (var buttonNo in buttons) {
                thiz.disableButton(buttonNo);
            }
        },

        hideAllButtons : function() {
            for (var buttonNo in buttons) {
                thiz.hideButton(buttonNo);
            }
        },

        hideAll : function() {
            thiz.hideAllButtons();
            thiz.setMessage("");
            thiz.setStatusMessage("");
            thiz.hideEditBox();
            thiz.hideComboBox();
        },

        /**
         * Fills the textarea which can be copied to the clipboard
         */
        prepareClipboard : function(text) {
            $(clipboard).text(text);
        },

        /**
         * This copies the content of the textarea (filled by prepareClipboard) to the clipboard.
         * This function can only be called by a callback from a user interaction!!!
         *
         * // Deprecated. The copy operation is done by the native backend since V3.1.
         */
        copyToClipboard : function() {

            var content = $(clipboard).text();
            if ((!content) || (content.length === 0)) {
                return;
            }

            // Chrome: Element must be visible when copying
            $(clipboard).show();

            try {
                $(clipboard).select();
                var successful = document.execCommand('copy');
                logDebug('Copying text was ' + successful);
            }
            catch (excption) {
                logDebug("Copying to clipboard failed");
            }

            $(clipboard).hide();
        },

        // Replaces the whole page with the given html code
        finishProcessing : function(url, target, pageContent, errorMessage) {
            
            var eventDetail = {};
            if (errorMessage) {
            	eventDetail.isOk = false;
            	eventDetail.message = errorMessage;
            } else {
            	eventDetail.isOk = true;
            	eventDetail.message = "OK";
            }
            if (typeof cloneInto !== "undefined") {
                // siehe https://stackoverflow.com/questions/18744224/triggering-a-custom-event-with-attributes-from-a-firefox-extension
            	eventDetail = cloneInto(eventDetail, document.defaultView);
            }
            var event = new CustomEvent("travic-sign-finished", {"bubbles": true, "detail" : eventDetail});
            dispatchEventOnPageElement(event);                                 
            
            if (thiz.hasModeFlag("DisablePageSubstitution")) {
              	logDebug("Skipping replacing page with content because of modeFlag 'DisablePageSubstitution'");
              	
            } else {
                logDebug("Replacing page with content");
                
                var targetDocument;
                var targetIframe;            
                if (target) {
                    var targetValue = target.trim().toLowerCase();
                    
                    // Ignore special targets _blank and _self
                    if ((targetValue == "_blank") || (targetValue == "_self")) {
                        targetDocument = document;
                        targetIframe = "";
                    }
                    else if (targetValue == "_parent") {
                        targetDocument = parent.document;
                        targetIframe = "";
                    }
                    else if (targetValue == "_top") {
                        targetDocument = top.document;
                        targetIframe = "";
                    }
                    else {
                        targetDocument = top.document;
                        targetIframe = target.trim();
                    }
                    
                    if (targetIframe) {
                        logDebug("Using target: " + targetIframe);
                    }
                }
                else {
                    targetDocument = document;
                    targetIframe = "";
                }
                            
                // Change URL in address bar for top level targets
                if (url && (!targetIframe) && (targetDocument === top.document)) {
                    logDebug("Updating URL in address bar to: " + url);
                    try {
                        history.replaceState(null, "", url);
                    } catch(err) {
                        logError("Cannot update history for URL: " + url);
                    }
                }
                else {
                    logDebug("Skipping update of URL in address bar because of non top-level target or iframe");
                }
                
                // Create a temporary html node
                var tempHtmlNode = targetDocument.createElement("html");
                tempHtmlNode.innerHTML = pageContent;
                
                var targetHead = $("head", targetDocument);
                var targetBody = $("body", targetDocument);
                if (targetIframe) {
                    var iFrame = findIframe(targetIframe, targetDocument);
                    if (iFrame) {
                        targetHead = iFrame.contents().find("head");
                        targetBody = iFrame.contents().find("body");
                    }
                }
                
                var tempHead = $(tempHtmlNode).find("head").first();
                if (tempHead) {
                    targetHead.html(tempHead.html());
                }
                
                var tempBody = $(tempHtmlNode).find("body").first();
                if (tempBody) {
                    targetBody.html(tempBody.html());
                }
            }
            
            thiz.closeGui();
        },

        stopAllAnimations : function() {
            thiz.stopStatusMessageAnimation();
        },

        ///////////////////////////////////////////////////
        // GUI handler functions
        ///////////////////////////////////////////////////

        setButtonProperty : function(buttonNo, name, value) {
            buttons[buttonNo][name] = value;
        },

        handleDefaultButtonClicked : function() {
            var buttonNo = thiz.getDefaultButton();
            thiz.handleButtonClicked(buttonNo);
        },

        handleButtonClicked : function(buttonNo) {
            // Handle special clipboard button
            // TODO This is deprecated. The copy operation is done by the native backend since V3.1.
            if (buttons[buttonNo].data("name") == "copy") {
                thiz.copyToClipboard();
            }
            
            if ((buttonNo >= 0) && (buttonNo < buttons.length)) {
                // Disable all buttons to avoid double clicks
                thiz.disableAllButtons();

                // Send event to native host
                guiProtocolService.handleButtonClick(buttonNo);
            }
        },

        handleNativeHostInfo : function(nativeHostVersion) {
            logDebug("handleNativeHostInfo - Version: " + nativeHostVersion); 
          
            // We use plain old Javascript here and not JQuery, because JQuery seems to skip the detail information.
            // Warning: Firefox doesn't transmit event details from extension to page!!
            var eventData = {"extensionVersion":  versionInfoService.getVersion(),            
                             "nativeHostVersion": nativeHostVersion};

            var event = new CustomEvent("travic-sign-completely-installed", { "bubbles": true, "detail": eventData});
            dispatchEventOnPageElement(event);                                 
        },
                
        ///////////////////////////////////////////////////////////////
        // Miscellaneous
        ///////////////////////////////////////////////////////////////

        getParameter : function(name) {
            return $(mainDiv).data(name.toLowerCase());
        },

        hasModeFlag : function(name) {
            var modeFlagsString = thiz.getParameter("ModeFlags");
            if (modeFlagsString) {
                var modeFlags = modeFlagsString.split(",");

                for (var index in modeFlags) {
                    modeFlags[index] = modeFlags[index].toLowerCase().trim();
                }

                return (modeFlags.indexOf(name.toLowerCase()) >= 0);
            }

            return false;
        },
        
        ///////////////////////////////////////////////////
        // Initialization 
        ///////////////////////////////////////////////////
        
        /**
         * Starts the gui if plugin invocation is on the current page.
         */
        startGuiIfAvailable : function() {
            logDebug("Searching for plugin on page..");
            
            var pluginDiv = $("div#" + PRODUCT_ID + "[data-type]:visible:not(.travic-sign-started)");
            if (pluginDiv.length) {
                initialize(pluginDiv);
            }
        },           
        
        start : function() {           
            if (isFirefox() && !isFirefoxVersionSufficient()) {
                logDebug("Firefox version insufficient. At least version " + getMinRequiredFirefoxVersion() + " is required.");
                return;
            }
        
            // Start GUI if already visible
            thiz.startGuiIfAvailable();

            // Starts an obverver to wait for new plugins added to the DOM.
            startPluginObserver();
                        
            // Register a global event listener that starts the GUI on request
            window.addEventListener("message", function(event) {
                if (event.source != window) {
                    return;
                }

                if (event.data && (event.data === "travic-sign-start")) {
                    thiz.startGuiIfAvailable();
                }
            });
        }        
        
    };
    return thiz;
})();

if (isSafari()) {
    document.addEventListener("DOMContentLoaded", function(event) {
        versionInfoService.init();
        gui.start();
    });
} else {
    $(document).ready(function() {
        gui.start();
    });
}
