"use strict";

if (typeof getNativeMessagingHostId === "undefined") {
    // functions from util.js are missing in chrome browser with manifest V3
    // importScripts does not work with firefox manifest
    importScripts("util.js")
}

chrome.runtime.onConnect.addListener(function(port) {
    var messagingService = newNativeMessageProxy(port);
    messagingService.initialize();
});

/**
 * This class forwards messages between content scripts and native host
 *
 * @param port Content port
 */
function newNativeMessageProxy(port) {
    
    var contentPort = port;

    // Enable/disable debug log
    var debug = false;

    var nativePort = null;   
    var tabId = null;

    function sendContentMessage(message) {
        if (debug === true) {
            logDebug("Sending message to content: " + JSON.stringify(message));
        }
        contentPort.postMessage(message);
    }

    function onContentMessageReceived(message) {
        if (debug === true) {
            logDebug("Receiving message from content script: " + JSON.stringify(message));
        }
        sendNativeMessage(message);    
    }

    function onContentPortDisconnect() {
        logDebug("Content port disconnected");

        if (chrome.runtime.lastError) {
            logError("Error: " + chrome.runtime.lastError.message);
        }

        hidePageIcon();
        nativePort.disconnect();    
    }

    function sendNativeMessage(message) {
        if (debug === true) {
            logDebug("Sending message to native host: " + JSON.stringify(message));
        }
        nativePort.postMessage(message);
    }

    function onNativeMessageReceived(message) {
        if (debug === true) {
            logDebug("Message from native host received: " + JSON.stringify(message));
        }
        sendContentMessage(message);
    }

    function onNativePortDisconnect() {
        logDebug("Native port disconnected");
        
        if (chrome.runtime.lastError) {
            logError("Error: " + chrome.runtime.lastError.message);
        }
        
        hidePageIcon();
        contentPort.disconnect();        
    }

    function logDebug(message) {
        if (debug === true) {
            var dateString = formatDateString(new Date());
            console.debug(dateString + " - " + PRODUCT_NAME + " [NativeMessageProxy]: " + message);
        }
    }

    function logError(message) {
        var dateString = formatDateString(new Date());
        console.error(dateString + " - " + PRODUCT_NAME + " [NativeMessageProxy]: " + message);
    }    

    function showPageIcon() {
        if (tabId && chrome.pageAction && chrome.pageAction.show) {
            chrome.pageAction.show(tabId);
        }
    }

    function hidePageIcon() {
        if (tabId && chrome.pageAction && chrome.pageAction.hide) {
            chrome.pageAction.hide(tabId);
        }
    }
    
    return {
        initialize : function() {
            logDebug("Initializing...");
            
            if ((contentPort.sender) && (contentPort.sender.tab) && (contentPort.sender.tab.id)) {
                tabId = contentPort.sender.tab.id;
            }
            
            // Initialize connection to content script
            contentPort.onMessage.addListener(onContentMessageReceived);
            contentPort.onDisconnect.addListener(onContentPortDisconnect);

            // Initialize connection to native host
            logDebug("Connecting to native messaging host '" + getNativeMessagingHostId() + "' ...");
            nativePort = chrome.runtime.connectNative(getNativeMessagingHostId());            
            nativePort.onDisconnect.addListener(onNativePortDisconnect);
            nativePort.onMessage.addListener(onNativeMessageReceived);

            showPageIcon();
           
            logDebug("Initialized");
        }
    };
}    
