"use strict";

/**
 * This 'class' is used to send/receive messages to/from a background script.
 *
 * @param onMessageReceivedCB Callback function which is called when a message is received
 * @param onConnectCB         Callback function which is called when the connection has been established.
 * @param onDisconnectCB      Callback function which is called when the connection gets closed.
 */
function newMessageService(onMessageReceivedCB, onConnectCB, onDisconnectCB) {
    
    var onMessageReceivedCallback = onMessageReceivedCB;
    var onConnectCallback = onConnectCB;
    var onDisconnectCallback = onDisconnectCB;

    var port = null;
    var inputMessageCounter = 0;
    
    var keepAliveTimer = null; // keep alive for Safari App Extension
    
    /**
     * Called when connection has been established.
     */
    function onConnect() {
        logDebug("Connection to background script has been established");
        
        if (onConnectCallback) {
            onConnectCallback();
        }
    }

    /**
     * Called when JSON message has been received.
     */
    function onMessageReceived(message) {
        if (optionsService.isDebugEnabled()) {
            var msgText = JSON.stringify(message);
            var msgLength = msgText.length;
            if (msgLength > 1000) {
                msgText = msgText.substring(0, 1000) + "...";
            }
            logDebug("Message received: " + msgText + " (" + msgLength + " Byte)");
        }

        if (onMessageReceivedCallback) {
            onMessageReceivedCallback(message);
        }
    }

    /**
     * Called when connection has been disconnected.
     */
    function onDisconnect() {
        logDebug("Connection disconnected");
        
        if (onDisconnectCallback) {
            onDisconnectCallback();
        }
    }

    /**
     * Prints debug output
     */
    function logDebug(message) {
        if (optionsService.isDebugEnabled()) {
            var dateString = formatDateString(new Date());
            var logMessage = message.replace(new RegExp("editBoxText.*", "i"), "editBoxText...");
            console.debug(dateString + " - " + PRODUCT_NAME + " [MessageService]: " + logMessage);
        }
    }

    /**
     * Prints error messages
     */
    function logError(message) {
        var dateString = formatDateString(new Date());
        var logMessage = message.replace(new RegExp("editBoxText.*", "i"), "editBoxText...");
        console.error(dateString + " - " + PRODUCT_NAME + " [MessageService]: " + logMessage);
    }
    
    
    /**
     * Keep alive for Safari App Extension.
     */
    function sendKeepAlive() {
        safari.extension.dispatchMessage("travic-sign-keepalive", { _connectionId : port });
    }

    function doConnect() {
        if (!port) {
            logDebug("Connecting to background script");
            
            if (isSafari()) {
                var idBytes = new Uint8Array(16);
                window.crypto.getRandomValues(idBytes);
                var idString = "";
                for (var i = 0; i < 16; ++i) {
                    idString += ("0" + idBytes[i].toString(16)).slice(-2);
                }
                logDebug("id: " + idString);
                port = idString;
                inputMessageCounter = 0;
                safari.self.addEventListener("message", function(event) {
                    if (event.name == "travic-sign-outputmessage") {
                        var msgObj = event.message;
                        logDebug("----------- RECEIVED: " + JSON.stringify(msgObj));
                        onMessageReceived(JSON.parse(msgObj._message));
                    }
                });
                safari.extension.dispatchMessage("travic-sign-connect", { _connectionId : idString });
                keepAliveTimer = setInterval(sendKeepAlive, 5000);
            } else {
                // Open port to background script
                port = chrome.runtime.connect();
                
                // Register event handlers
                port.onMessage.addListener(onMessageReceived);
                port.onDisconnect.addListener(onDisconnect);
            }
            onConnect();
        }
    }
    
    ///////////////////////////////////////////////////////////////
    // Public interface
    ///////////////////////////////////////////////////////////////
    
    return {

        /**
         * Connects to native host / background script
         */
        connect : function() {
            doConnect();
        },

        /**
         * Disconnect from native host / background script
         */
        disconnect : function() {
            if (port) {
                logDebug("Disconnecting");
                
                if (isSafari()) {
                    if (keepAliveTimer) {
                        clearInterval(keepAliveTimer);
                        keepAliveTimer = null;
                    }
                    inputMessageCounter += 1;
                    safari.extension.dispatchMessage("travic-sign-disconnect", { _connectionId : port,
                                                                                 _messageId    : inputMessageCounter});
                } else {
                    port.disconnect();
                }
                port = null;
            }
        },
        
        /**
         * Sends a JSON message to native host / background script
         */
        sendMessage : function(message) {
            if (!port) {
                doConnect();
            }
            if (optionsService.isDebugEnabled()) {
                logDebug("Sending message: " + JSON.stringify(message));
            }
            if (isSafari()) {
                inputMessageCounter += 1;
                safari.extension.dispatchMessage("travic-sign-inputmessage", { _connectionId : port,
                                                                               _messageId    : inputMessageCounter,
                                                                               _message      : JSON.stringify(message)});
            } else {
                port.postMessage(message);
            }
        }
        
    };

}
