"use strict";

/**
 * This 'class' creates an animation for a message. A number of characters is appended to the message
 * until it reaches the maximum size. When it does the animation restarts with the initial message.
 * The message is updated in the given interval.
 *
 * @param elementParam        JQuery element to animate
 * @param messageParam        Initial text message
 * @param characterParam      Character used for animation
 * @param millisecondsParam   Update interval (in milliseconds) 
 */
function newMessageAnimation(elementParam, messageParam, characterParam, millisecondsParam) {
    
    var element = elementParam;
    var initialMessage = messageParam;
    var character = characterParam;
    var milliseconds = millisecondsParam;

    var interval = null;
    var running = false;
    var iterationNo = 0;    
    var currentMessage = initialMessage;

    // Enable/disable debug log
    var debug = false;
    
    /**
     * Internal function to process the message update
     */
    function process() {
        iterationNo++;
        currentMessage = currentMessage + character;

        updateMessageText(currentMessage);

        var hasOverflow = element[0].offsetWidth < element[0].scrollWidth;
        if (hasOverflow) {
            currentMessage = initialMessage;
            updateMessageText(currentMessage);
        }
    }

    /**
     * Updates the message text
     */
    function updateMessageText(message) {
        $(element).text(message);
    }

    function logDebug(message) {
        if (debug === true) {
            var dateString = formatDateString(new Date());
            console.debug(dateString + " - " + PRODUCT_NAME + " [MessageAnimation]: " + message);
        }
    }

    function logError(message) {
        var dateString = formatDateString(new Date());
        console.error(dateString + " - " + PRODUCT_NAME + " [MessageAnimation]: " + message);
    }

   
    ///////////////////////////////////////////////////////////////
    // Public interface
    ///////////////////////////////////////////////////////////////
    
    var thiz = {
        
        /**
         * Starts the animation
         */
        start : function() {
            if (running) {
                thiz.stop();
            }

            logDebug("Starting animation...");

            currentMessage = initialMessage;
            iterationNo = 0;
            running = true;

            updateMessageText(initialMessage);
            interval = setInterval(process, milliseconds);
        },

        /**
         * Stops the animation
         */
        stop : function() {
            if (running) {                
                logDebug("Stopping animation");
                    
                updateMessageText(initialMessage);
                clearInterval(interval);
            }
        },

        /**
         * Returns whether this animation is running
         */
        isRunning : function() {
            return running;
        }
        
    };
    
    return thiz;
}   