"use strict";

/**
 * Communication with background script.
 *
 * @param guiParam A gui instance
 */
function newGuiProtocolService(guiParam, parameters, onConnectCB, onDisconnectCB, onFirstNativeMessageCB) {
    
    var gui = guiParam;
    var pluginParameters = parameters;                       
    var onConnectCallback = onConnectCB;
    var onDisconnectCallback = onDisconnectCB;    
    var onFirstNativeMessageCallback = onFirstNativeMessageCB;
    
    var pluginMainStarted = false;
    
    var messageService = newChunkedMessageService(onMessageReceived,
                                                  onConnect,
                                                  onDisconnect);
                                                   
    var nativeMessageReceived = false;                                               
                                                       
    //////////////////////////////////////////////////////////////////////////////
    // Technical functions to handle connection to background script / native host
    //////////////////////////////////////////////////////////////////////////////
    
    // Sends a message to background script / native host
    function sendMessage(message) {
        if (optionsService.isDebugEnabled()) {
            logDebug("Sending message to background script: " + JSON.stringify(message));
        }
        messageService.sendMessage(message);
    }

    // Receives a message from background script / native host
    function onMessageReceived(message) {
        if (optionsService.isDebugEnabled()) {
            var msgText = JSON.stringify(message);
            var msgLength = msgText.length;
            if (msgLength > 500) {
                msgText = msgText.substring(0, 500) + "...";
            }
            logDebug("Receiving message from background script: " + msgText + " (" + msgLength + " Byte)");
        }

        // Check for first incoming native message
        if ((nativeMessageReceived === false) && onFirstNativeMessageCallback) {
            logDebug("First message from background script received");
            onFirstNativeMessageCallback();
        }
        nativeMessageReceived = true;
        
        if (("eventName" in message) && ("eventData" in message)) {
            processNativeEvent(message.eventName, message.eventData);
        }
    }

    // Connects to background script / native host 
    function connect() {
        logDebug("Connecting to background script");        
        messageService.connect();
    }
    
    // Called when connection to background script / native host has been established
    function onConnect() {
        logDebug("Connection to background script has been established");
        
        if (onConnectCallback) {
            onConnectCallback();
        }
     }

    // Disconnects from background script / native host 
    function disconnect() {
        logDebug("Disconnecting from background script");
                
        // Disconnect connection
        messageService.disconnect();
    }
    
    // Called when connection to background script / native host gets closed
    function onDisconnect(message) {
        logDebug("Connection to background script disconnected");

        if (onDisconnectCallback) {
            onDisconnectCallback();
        }
    }


    ///////////////////////////////////////////////////////////////
    // Functions GUI -> native host
    ///////////////////////////////////////////////////////////////

    function handleButtonClick(buttonNo) {
        logDebug("handleButtonClick: " + "button" + buttonNo);

        var event = {"buttonName" : "button" + buttonNo};

        if (gui.isEditBoxVisible()) {
            event.editBoxText = gui.getEditBoxText();
        }

        if (gui.isComboBoxVisible()) {
            event.comboBoxText = gui.getSelectedComboboxText();
            event.comboBoxIndex = gui.getSelectedComboboxIndex();
        }

        sendGuiEvent("buttonClick", event);
    }      
        
    // Sends an event to the background script
    function sendGuiEvent(name, data) {
        var guiStatus = {"height" : gui.getHeight(),
                         "width"  : gui.getWidth()};

        var message = {"eventName" : name,
                       "eventData" : data,
                       "guiStatus" : guiStatus};

        sendMessage(message);
    }

    // Sends a 'startPlugin' event to background script if not yet done
    function startPluginMain() {
        if (pluginMainStarted === true) {
            logDebug("startPluginMain: PluginMain already started");
            return;
        }
        
        // Build list of all plugin parameters
        var params = {};
        $.each(parameters, function(name, value) {
            params[name] = "" + value; // Always use Strings!!!
        });

        // Add additional parameters
        var extensionVersion = {"version" : versionInfoService.getVersion()};
        var eventData = {"inputParameters" : params,
                         "userAgent"       : navigator.userAgent,
                         "extension"       : extensionVersion,
                        };

        sendGuiEvent("startPlugin", eventData);
        
        pluginMainStarted = true;
    }

    // Sends a 'stopPlugin' event to background script if not yet done
    function stopPluginMain() {
        if (pluginMainStarted === false) {
            logDebug("stopPluginMain: Plugin main NOT started");
            return;
        }        
        
        // Send stop message
        var eventData = {};
        
        try {
            sendGuiEvent("stopPlugin", eventData);
        }
        catch(exception) {
            logError("Error sending 'stopPlugin' event. Maybe connection has been disconnected.");
        }

        pluginMainStarted = false;
    }
    
    // Sends a "getPluginInfo" event to background script
    function requestNativeHostVersion() {
        var message = {"eventName" : "getNativeHostInfo",
                       "eventData" : null};
        sendMessage(message);        
    }
    
    ///////////////////////////////////////////////////////////////
    // Functions Native host -> GUI
    ///////////////////////////////////////////////////////////////

    // Processes GUI events received from native host
    function processNativeEvent(eventName, eventData) {
        if (optionsService.isDebugEnabled()) {
            var eventText = JSON.stringify(eventData);
            var eventLength = eventText.length;
            if (eventLength > 1000) {
                eventText = eventText.substring(0, 1000) + "...";
            }            
            logDebug("Processing event from native messaging host: '" + eventName + "' with data: " + eventText);
        }

        // Stop all running animations
        gui.stopAllAnimations();

        if (eventName == "sendHttpGetRequest") {
            processNativeHttpGetEvent(eventData);
        }
        else if (eventName == "sendHttpPostRequest") {
            processNativeHttpPostEvent(eventData);
        }
        else if (eventName == "sendHttpPostXmlRequest") {
            processNativeHttpPostXmlEvent(eventData);
        }
        else if (eventName == "clipboard") {
            processNativeClipboardEvent(eventData);
        }
        else if (eventName == "guiSetup") {
            processNativeGuiSetupEvent(eventData);
        }
        else if (eventName == "nativeHostInfo") {
            processNativeHostInfoEvent(eventData);
        }
        else if (eventName == "setControls") {
            if ("message" in eventData) {
                processNativeMessageEvent(eventData.message);
            }
            if ("statusMessage" in eventData) {
                processNativeStatusMessageEvent(eventData.statusMessage);
            }
            if ("button1" in eventData) {
                processNativeButtonEvent(1, eventData.button1);
            }
            if ("button2" in eventData) {
                processNativeButtonEvent(2, eventData.button2);
            }
            if ("button3" in eventData) {
                processNativeButtonEvent(3, eventData.button3);
            }
            if ("editBox" in eventData) {
                processNativeEditBoxEvent(eventData.editBox);
            }
            if ("comboBox" in eventData) {
                processNativeComboBoxEvent(eventData.comboBox);
            }

            gui.resize();
        }
    }

    // Processes a HTTP GET event
    function processNativeHttpGetEvent(event) {
        logDebug("Process native HTTP GET event.");
        gui.disableAllButtons();

        var target               = "target"               in event ? event.target                : null;
        var url                  = "url"                  in event ? event.url                  : null;
        var additionalHeaderLine = "additionalHeaderLine" in event ? event.additionalHeaderLine : null;

        sendHttpGetRequest(target,
                                url,
                                additionalHeaderLine);
    }

    // Processes a HTTP POST event
    function processNativeHttpPostEvent(event) {
        logDebug("Process native HTTP POST event.");
        gui.disableAllButtons();

        var target               = "target"               in event ? event.target                  : null;
        var url                  = "url"                  in event ? event.url                     : null;
        var contentType          = "contentType"          in event ? event.contentType             : null;
        var data                 = "data"                 in event ? decodeBase64(event.data)      : null;
        var additionalHeaderLine = "additionalHeaderLine" in event ? event.additionalHeaderLine    : null;

        sendHttpPostRequest(target,
                            url,
                            data,
                            contentType,
                            additionalHeaderLine);
    }

    // Processes a HTTP POST XML event
    function processNativeHttpPostXmlEvent(event) {
        logDebug("Process native HTTP POST XML event.");
        gui.disableAllButtons();

        var url                  = "url"                  in event ? event.url                     : null;
        var contentType          = "contentType"          in event ? event.contentType             : null;
        var data                 = "data"                 in event ? event.data                    : null;
        var additionalHeaderLine = "additionalHeaderLine" in event ? event.additionalHeaderLine    : null;

        sendHttpPostXmlRequest(url,
                               data,
                               contentType,
                               additionalHeaderLine);
    }

    // Decodes a base64 encoded string and returns a blob
    function decodeBase64(base64EncodedString) {
        var binary = atob(base64EncodedString);
        var array = new Uint8Array(binary.length);
        for( var i = 0; i < binary.length; i++ ) { array[i] = binary.charCodeAt(i); }
        return new Blob([array]);
    }

    // Prepares copying to clipboar by filling
    // a textarea which can be copyied by the user
    function processNativeClipboardEvent(event) {
        if ("text" in event) {
            gui.prepareClipboard(event.text);
        }
    }

    // Processes message updates
    function processNativeMessageEvent(event) {
        if ("text" in event) {
            if (("error" in event) && (event.error === true)) {
                gui.setErrorMessage(event.text);
            }
            else {
                gui.setMessage(event.text);
            }
        }
        if ("fontFamily" in event) {
            gui.setMessageFontFamily(event.fontFamily);
        }
    }

    // Processes status message updates
    function processNativeStatusMessageEvent(event) {
        if ("text" in event) {
            gui.setStatusMessage(event.text);

            // Maybe the status message should be animated
            if ("animation" in event) {
                var animation = event.animation;

                var character = ".";
                if ("character" in animation) {
                    character = animation.character;
                }

                var interval = 200;
                if ("interval" in animation) {
                    interval = animation.interval;
                }

                gui.startStatusMessageAnimation(event.text, character, interval);
            }
        }

    }

    // Processes button updates
    function processNativeButtonEvent(buttonNo, event) {
        if (optionsService.isDebugEnabled()) {
            logDebug("Process native event for button: " + buttonNo + ", with data: " + JSON.stringify(event));
        }

        // Enable/disable button
        if ("enable" in event)  {
            if (event.enable) {
                gui.enableButton(buttonNo);
            }
            else {
                gui.disableButton(buttonNo);
            }
        }

        if ("visible" in event) {
            if (event.visible) {
                gui.showButton(buttonNo);
            }
            else {
                gui.hideButton(buttonNo);
            }
        }

        // Set button text
        if ("text" in event) {
            gui.setButtonText(buttonNo, event.text);
        }

        // Set button name
        if ("name" in event) {
            gui.setButtonName(buttonNo, event.name);
        }

        // Set button focus
        if (("focus" in event) && (event.focus)) {
            gui.focusButton(buttonNo);
        }

        // Set default button
        if (("default" in event) && (event["default"])) {
            gui.setDefaultButton(buttonNo);
        }

        // Special button functions

        gui.setButtonProperty(buttonNo, "clipboard", false);
        if ("clipboard" in event) {
            gui.setButtonProperty(buttonNo, "clipboard", event.clipboard);
        }
    }

    function processNativeEditBoxEvent(event) {
        if (optionsService.isDebugEnabled()) {
            logDebug("Process native event for editBox with data: " + JSON.stringify(event));
        }

        if ("visible" in event) {
            if (event.visible) {
                gui.showEditBox();
            }
            else {
                gui.hideEditBox();
            }
        }

        if (("focus" in event) && (event.focus)) {
            gui.focusEditBox();
        }

        gui.setEditBoxText("");
        if ("text" in event) {
            gui.setEditBoxText(event.text);
        }

        gui.setEditBoxPlaceholder("");
        if ("placeholder" in event) {
            gui.setEditBoxPlaceholder(event.placeholder);
        }

        // Set name of edit box
        gui.setEditBoxName("");
        if ("name" in event) {
            gui.setEditBoxName(event.name);
        }

        gui.clearEditBoxSize();
        if ("charWidth" in event) {
            gui.setEditBoxSize(event.charWidth);
        }

        if ("label" in event) {
            gui.setStatusMessage(event.label);
        }

        gui.setEditBoxTypeClear();
        if (("type" in event) && (event.type == "pin") ){
            gui.setEditBoxTypePassword();
        }
    }

    function processNativeComboBoxEvent(event) {
        if (optionsService.isDebugEnabled()) {
            logDebug("Process native event for comboBox with data: " + JSON.stringify(event));
        }

        if ("visible" in event) {
            if (event.visible) {
                gui.showComboBox();
            }
            else {
                gui.hideComboBox();
            }
        }

        if (("focus" in event) && (event.focus)) {
            gui.focusComboBox();
        }

        // Update entries in combobox
        gui.clearComboBox();
        if ("entries" in event) {
            for (var index in event.entries) {
                var entry = event.entries[index];
                if (entry.hasOwnProperty("text") && (entry.hasOwnProperty("value"))) {
                    gui.addComboBoxEntry(entry.text, entry.value);
                }
            }
         }
    }

    function processNativeGuiSetupEvent(event) {
        if (optionsService.isDebugEnabled()) {
            logDebug("Process native event for GUI setup with data: " + JSON.stringify(event));
        }

        if ("standardButtonWidthText" in event) {
            gui.calculateAndSetMinimalButtonWidth(event.standardButtonWidthText);
        }
    }

    // Processes event 'nativeHostInfo'
    function processNativeHostInfoEvent(event) {
        gui.handleNativeHostInfo(event.version);
    }
    
    ///////////////////////////////////////////////////
    // Communication with web server
    ///////////////////////////////////////////////////

    function sendHttpGetRequest(target, url, additionalHeaderLine) {
        url = buildAbsoluteUrlOnCurrentLocation(url);
        logDebug("Sending HTTP GET request to URL: " + url);

        gui.disableAll();

        // Create own XHR. JQuery uses a different jqXHR which doesn't contain all necessary informations, 
        // e.g. it does not include the reponseURL.
        var xhr = $.ajaxSettings.xhr();
        xhr.withCredentials = true;
        
        $.ajax({
            type        : "GET",
            url         : url,
            xhr         : function() {
                              // Use our instance of XHR
                              return xhr;
                          },
            xhrFields: { withCredentials: true },
            crossDomain: true,
            beforeSend : function (request) {
                if (additionalHeaderLine) {
                    var pos = additionalHeaderLine.indexOf(":");
                    if (pos >= 0) {
                        var fieldName = additionalHeaderLine.substring(0, pos).trim();
                        var fieldValue = additionalHeaderLine.substring(pos + 1).trim();
                        request.setRequestHeader(fieldName, fieldValue);
                    }
                }
            },
            success : function(data, textStatus, jqXHR) {
                logDebug("HTTP GET request finished successfully, response data: " + data);
                disconnect();
                
                gui.finishProcessing(xhr.responseURL, target, data);
            },
            error : function(jqXHR, textStatus, errorThrown) {
                var errorMsg = "HTTP GET request failed: " + errorThrown + " (" + jqXHR.status + ")";
                logError(errorMsg);
                disconnect();
                
                if (!jqXHR.responseType || jqXHR.responseType === "text") {
                    gui.finishProcessing(xhr.responseURL, target, jqXHR.responseText, errorMsg);
                }
                else {
                    gui.setMessage(errorMsg);
                }

            }
        });
    }

    function sendHttpPostRequest(target, url, data, contentType, additionalHeaderLine) {
        url = buildAbsoluteUrlOnCurrentLocation(url);
        logDebug("Sending HTTP POST request to URL: " + url);

        gui.disableAll();
        
        // Create own XHR. JQuery uses a different jqXHR which doesn't contain all necessary informations, 
        // e.g. it does not include the reponseURL.
        var xhr = $.ajaxSettings.xhr();
        xhr.withCredentials = true;        
        
        $.ajax({
            type        : "POST",
            url         : url,
            data        : data,
            contentType : contentType,
            processData : false,
            cache       : false,
            dataType    : "html",
            xhrFields: { withCredentials: true },
            crossDomain: true,
            xhr         : function() {
                              // Use our instance of XHR
                              return xhr;
                          },                          
            beforeSend: function (request) {
                if (additionalHeaderLine) {
                    var pos = additionalHeaderLine.indexOf(":");
                    if (pos >= 0) {
                        var fieldName = additionalHeaderLine.substring(0, pos).trim();
                        var fieldValue = additionalHeaderLine.substring(pos + 1).trim();
                        request.setRequestHeader(fieldName, fieldValue);
                    }
                }
            },
            success     : function(data, textStatus, jqXHR) {
                logDebug("HTTP POST request finished successfully, response data: " + data);
                disconnect();
                
                gui.finishProcessing(xhr.responseURL, target, data);
            },
            error       : function(jqXHR, textStatus, errorThrown) {
                var errorMsg = "HTTP POST request failed: " + errorThrown + " (" + jqXHR.status + ")";
                logError(errorMsg);
                disconnect();

                if (!jqXHR.responseType || jqXHR.responseType === "text") {
                    gui.finishProcessing(xhr.responseURL, target, jqXHR.responseText, errorMsg);
                }
                else {
                    gui.setErrorMessage(errorMsg);
                }
            }
        });
    }

    function sendHttpPostXmlRequest(url, data, contentType, additionalHeaderLine) {
        url = buildAbsoluteUrlOnCurrentLocation(url);
        logDebug("Sending HTTP POST XML request to URL: " + url);
        
        $.ajax({
            type        : "POST",
            url         : url,
            data        : data,
            contentType : "application/xml",
            processData : false,
            cache       : false,
            dataType    : "xml",
            beforeSend: function (request) {
                if (additionalHeaderLine) {
                    var pos = additionalHeaderLine.indexOf(":");
                    if (pos >= 0) {
                        var fieldName = additionalHeaderLine.substring(0, pos).trim();
                        var fieldValue = additionalHeaderLine.substring(pos + 1).trim();
                        request.setRequestHeader(fieldName, fieldValue);
                    }
                }
            },
            success     : function(xml, textStatus, jqXHR) {
                var eventData = {"data"       : new XMLSerializer().serializeToString(xml),
                                 "status"     : jqXHR.status,
                                 "textStatus" : textStatus};
                sendGuiEvent("receiveHttpPostXmlResponse", eventData);
            },
            error       : function(jqXHR, textStatus, errorThrown) {
                logError("HTTP POST XML request failed: " + errorThrown + " (" + jqXHR.status + ")");
                var eventData = {"data"        : data,
                                 "status"      : jqXHR.status,
                                 "textStatus"  : textStatus,
                                 "errorThrown" : errorThrown};
                sendGuiEvent("receiveHttpPostXmlResponse", eventData);
            }
        });
    }
    
    ///////////////////////////////////////////////////////////////
    // Miscellaneous
    ///////////////////////////////////////////////////////////////

    // Sends an error message to the server
    function sendErrorMessage(returnCode, returnMessage) {
        logDebug("Sending error message to server: " + returnMessage);

        gui.disableAll();

        var target = gui.getParameter("target");
        var url = gui.getParameter("returnURL");
        var contentType = "application/x-www-form-urlencoded";
        var additionalPostData = gui.getParameter("AdditionalPostData");
        var additionalHeaderLine = gui.getParameter("additionalHeaderLine");

        var data = "ReturnCode=" + encodeURI(returnCode) + "&" +
                   "ReturnMessage="+ encodeURI(returnMessage) + "&" +
                   additionalPostData;
        sendHttpPostRequest(target, url, data, contentType, additionalHeaderLine);
    }

    function logDebug(message) {
        if (optionsService.isDebugEnabled()) {
            var dateString = formatDateString(new Date());
            var logMessage = message.replace(new RegExp("editBoxText.*", "i"), "editBoxText...");
            console.debug(dateString + " - " + PRODUCT_NAME + " [guiProtocolService]: " + logMessage);
        }
    }

    function logError(message) {
        var dateString = formatDateString(new Date());
        var logMessage = message.replace(new RegExp("editBoxText.*", "i"), "editBoxText...");
        console.error(dateString + " - " + PRODUCT_NAME + " [guiProtocolService]: " + logMessage);
    }
    
    ///////////////////////////////////////////////////////////////
    // Public interface
    ///////////////////////////////////////////////////////////////
    
    return {
        
         // Connects to background script / native host.
        connect : connect,
        
         // Disconnects from background script / native host.
        disconnect : disconnect, 
        
        // Sends a 'startPlugin' event to background script to start execution 
        startPluginMain : startPluginMain,
        
        // Sends a 'stopPlugin' event to background script to stop execution 
        stopPluginMain : stopPluginMain,
        
        // Sends a 'buttonClick' event to background script
        handleButtonClick : handleButtonClick,
              
        // Request the version of native host  
        requestNativeHostVersion : requestNativeHostVersion,
        
        // Sends an error message to the calling server
        sendErrorMessage : sendErrorMessage
    };
}
