"use strict";

/**
 * As Chrome supports only max. 1MB per message, this 'class' 
 * support longer messages.
 * Therefore it splits the whole message in smaller chunks
 * and send them one by one.
 *
 * @param onMessageReceivedCB Callback function which is called when a message is received
 * @param onConnectCB         Callback function which is called when the connection has been established.
 * @param onDisconnectCB      Callback function which is called when the connection gets closed.
 */
function newChunkedMessageService(onMessageReceivedCB, onConnectCB, onDisconnectCB) {
    var onMessageReceivedCallback = onMessageReceivedCB;
    var onConnectCallback = onConnectCB;
    var onDisconnectCallback = onDisconnectCB;

    // Maximum length of chunks when sending (less than 1MB defined by Google because of escape characters)
    var maxChunkLength = 800 * 1024; // 800 KB

    // Enable/disable debug log
    var debug = false;

    // Message counter
    var messageNo = 0;

    var messageService = null;

    // Container for message receiving
    var lastReceivedMessage = "";
        
    function onConnect() {
        logDebug("Connection to background script has been established");
        
        if (onConnectCallback) {
            onConnectCallback();
        }
    }

    function onDisconnect() {
        logDebug("Connection disconnected");

        if (onDisconnectCallback) {
            onDisconnectCallback();
        }
    }

    function sendChunkedMessage(content) {
        // Increase message counter
        messageNo++;

        // Send message in chunks
        var pos = 0;
        while (pos < content.length) {
            var chunk = content.substring(pos, pos + maxChunkLength + 1);
            var lastChunk = (pos + chunk.length + 1 > content.length);

            var chunkMessage = {"transferMethod" : "chunked",
                                "msgNo"          : messageNo,
                                "lastChunk"      : lastChunk,
                                "content"        : chunk};
            messageService.sendMessage(chunkMessage);

            pos += chunk.length;
        }
    }

    function onMessageReceived(message) {
        if (debug === true) {
            var msgText = JSON.stringify(message);
            var msgLength = msgText.length;
            if (msgLength > 1000) {
                msgText = msgText.substring(0, 1000) + "...";
            }
            logDebug("Message chunk received: " + msgText + " (" + msgLength + " Byte)");
        }
        
        // Verify that this is really a chunked message.
        // If not the received message is already complete.
        if (!("transferMethod" in message) || 
            (message.transferMethod != "chunked")) {
            onMessageReceivedCallback(message);
            return;
        }

        // Parse chunked message

        var msgNo = -1;
        var lastChunk = false;
        var content = "";

        if ("msgNo" in message) {
            msgNo = message.msgNo;
            logDebug("MessageNo: " + msgNo);
            // TODO Verify messageNo+
        }

        if ("lastChunk" in message) {
            lastChunk = message.lastChunk;
            logDebug("LastChunk: " + lastChunk);
        }

        if ("content" in message) {
            content = message.content;
        }

        // Append chunk to already received chunks
        lastReceivedMessage += content;

        // Only call callback when all chunks have been received
        if (lastChunk) {
            logDebug("Last chunk received. Complete message length: " + lastReceivedMessage.length);
            
            onMessageReceivedCallback(JSON.parse(lastReceivedMessage));      
            lastReceivedMessage = "";
        }
    }

    function logDebug(message) {
        if (debug === true) {
            var dateString = formatDateString(new Date());
            console.debug(dateString + " - " + PRODUCT_NAME + " [ChunkedMessageService]: " + message);
        }
    }

    function logError(message) {
        var dateString = formatDateString(new Date());
        console.error(dateString + " - " + PRODUCT_NAME + " [ChunkedMessageService]: " + message);
    }
    
    ///////////////////////////////////////////////////////////////
    // Public interface
    ///////////////////////////////////////////////////////////////
    
    var thiz = {
        
        connect : function() {
            logDebug("Connecting to background script");
            
            if (messageService) {
                thiz.disconnect();
            }
            
            messageService = newMessageService(onMessageReceived, 
                                               onConnect,
                                               onDisconnect);
            messageService.connect();
        },

        disconnect : function() {
            logDebug("Disconnecting");
            
            if (messageService) {
                messageService.disconnect();
                messageService = null;
            }
        },

        sendMessage : function(message) {
            if (debug === true) {
                logDebug("Sending message: " + JSON.stringify(message));
            }

            if (!messageService) {
                return;
            }
            
            // Only large messages should be sent chunks.
            // For small messages this is too much overhead.
            var messageContent = JSON.stringify(message);
            if (messageContent.length > maxChunkLength) {
                sendChunkedMessage(messageContent);
            }
            else {
                messageService.sendMessage(message);
            }
        }
        
    };
    
    return thiz;
}


