"use strict";

var apiConverter = (function() {
    
    /**
     * Searches for <embed> tag and converts it to HTML5 API
     */
    function searchAndConvertEmbedTagToHtml5(message) {
        logDebug("Searching for <embed>...");

        var embedTagSelector = "embed[type^='application/x-ppi-security-plugin']";
        
        var embedTag = $(embedTagSelector);
        if (embedTag.length) {
            convertEmbedTagToHtml5(embedTag);
        }
    }

    /**
     * Convert <embed> tag to HTML5 API
     */
    function convertEmbedTagToHtml5(embedTag) {
        if (embedTag) {

            // Only evaluate visible elements
            if (!$(embedTag).is(":visible")) {
                logDebug("<embed> tag found on page. But it is NOT visible. Ignoring...");
                return;
            }

            // Check for supported type
            var type = $(embedTag).attr("type");
            logDebug("Instance with <embed> and type '" + type + "' found");
            if (!isExtensionTypeSupported(type)) {
                logError("Type '" + type + "' is not supported!");
                return;
            }

            logDebug("Plugin invocation with <embed> tag and type '" + type + "' found on page.");

            // Id which is used to trigger the extension.
            var productId;
            var productIdAttr = $(embedTag).attr('productId');
            if (typeof productIdAttr !== typeof undefined && productIdAttr !== false) {
                productId = $(embedTag).attr("productId");
                logDebug("Parameter 'productId' found: " + productId);
            }

            // Check for supported product id
            if (productId && (productId !== PRODUCT_ID)) {
                logDebug("productId '" + productId + "' not supported. Ignoring...");
                return;
            }

            logDebug("Converting plugin invocation to HTML5 API...");
            
            var pluginDiv = document.createElement("div");
            $(pluginDiv).attr("id", PRODUCT_ID);

            // Copy all parameters
            $(embedTag).each(function() {
                $.each(this.attributes, function() {
                    if (this.name === "id") {
                        // Ignore
                    }
                    else if (this.name === "class") {
                        logDebug.call(this, "Setting class attribute to: " + this.value);
                        $(pluginDiv).attr("class", this.value);
                    }
                    else if (this.name === "style") {
                        logDebug.call(this, "Setting style attribute to: " + this.value);
                        $(pluginDiv).attr("style", this.value);
                    }
                    else {
                        logDebug.call(this, "Setting Plugin-Parameter '" + this.name + "': " + this.value);

                        // Adjust old font size parameter for CSS
                        if ((this.name.toLowerCase() === "fontsize") && ($.isNumeric(this.value)) && (this.value >= 2)) {
                            this.value = this.value - 2;
                        }

                        $(pluginDiv).attr("data-" + this.name, this.value);
                    }
                });
            });

            $(pluginDiv).addClass(PRODUCT_ID);
            $(pluginDiv).addClass("travic-sign");

            // Copy content of embed tag
            $(embedTag).contents().clone().appendTo($(pluginDiv));
            
            // At last replace the embed tag
            $(embedTag).replaceWith(pluginDiv);
        }
    }

    /**
     * Maps Internet explorer classid to mime-type
     */
    function getTypeForClassId(classId) {
        if (!classId) {
            return undefined;
        }
        
        // Remove everything before :
        var index = classId.indexOf(":");
        if (index >= 0) {
            classId = classId.substr(index + 1);
        }

        for (var entry in CLASSID_TO_MIME_TYPE_MAP) {
            if (CLASSID_TO_MIME_TYPE_MAP.hasOwnProperty(entry)) {
                if (entry === classId) {
                    return CLASSID_TO_MIME_TYPE_MAP[classId];
                }
            }
        }

        return undefined;
    }

    /**
     * Searches for <object> tag and converts it to HTML5 API
     */
    function searchAndConvertObjectTagToHtml5() {
        logDebug("Searching for <object>...");

        var objectTagSelector = "object[type^='application/x-ppi-security-plugin'], object[classid]";
                
        // First try to find <object> with type definition
        var objectTag = $(objectTagSelector);
        if (objectTag.length) {
            convertObjectTagToHtml5(objectTag);
        }
    }

    /**
     * Convert <object> tag to HTML5 API
     */
    function convertObjectTagToHtml5(objectTag) {
        if (objectTag) {

            // Only evaluate visible elements
            if (!$(objectTag).is(":visible")) {
                logDebug("<object> found on page. But it is NOT visible. Ignoring...");
                return;
            }
        
            var type = $(objectTag).attr("type");
            if (!type || !isExtensionTypeSupported(type)) {                
                // If type is missing or invalid try classid instead
                var classId = $(objectTag).attr("classid");                
                if (classId) {
                    type = getTypeForClassId(classId);
                    if (type && isExtensionTypeSupported(type)) {
                        logDebug("<object> with classid '" + classId + "' found on page. Mapping to type '" + type + "'");
                    } else {    
                        logError("<object> with invalid classid '" + classId + "' found on page. Ignoring...");
                        return;
                    }
                }
                else {
                    logError("<object> with unknown type '" + type + "' found on page. Ignoring...");
                    return;
                }
            }
        
            // Check for supported type
            if (!isExtensionTypeSupported(type)) {
                logError("Type '" + type + "' is not supported!");
                return;
            }

            logDebug("Plugin invocation with <object> and type '" + type + "' found on page.");
            
            // Detect product Id which is used to trigger the extension. 
            var productId;
            $(objectTag).children("param").each(function() {
                var name = $(this).attr("name");
                if (name === "productId") {
                    productId = $(this).attr("value");
                    logDebug.call(this, "Parameter 'productId' found: " + productId);
                }
            });

            // Check for supported product id
            if (productId && (productId !== PRODUCT_ID)) {
                logDebug("productId '" + productId + "' not supported. Ignoring...");
                return;
            }
            
            logDebug("Converting plugin invocation to HTML5 API...");
            
            var pluginDiv = document.createElement("div");
            $(pluginDiv).attr("id", PRODUCT_ID);
            $(pluginDiv).attr("style", $(objectTag).attr("style"));
            $(pluginDiv).attr("class", $(objectTag).attr("class"));
            $(pluginDiv).addClass(PRODUCT_ID);
            $(pluginDiv).addClass("travic-sign");

            // Copy all parameters
            $(pluginDiv).attr("data-type", type);
            $(pluginDiv).attr("data-height", $(objectTag).attr("height"));
            $(pluginDiv).attr("data-width", $(objectTag).attr("width"));
            $(objectTag).children("param").each(function() {
                var name = $(this).attr("name");
                var value = $(this).attr("value");
                logDebug.call(this, "Setting Plugin-Parameter '" + name + "': " + value);

                // Adjust old font size parameter for CSS
                if ((name.toLowerCase() === "fontsize") && ($.isNumeric(value)) && (value >= 2)) {
                    value = value - 2;
                }

                $(pluginDiv).attr("data-" + name, value);
            });

            // Copy content of object tag (excluding params)          
            $(objectTag).contents().filter(function(){
                return !((this.nodeType == 1) && (this.nodeName.toLowerCase() == "param"));
            }).clone().appendTo($(pluginDiv));
            
            // At last replace the object tag
            $(objectTag).replaceWith(pluginDiv);
        }
    }
    
    /**
     * Registers a global event listener that starts the conversion process on request
     */
    function addGlobalEventListener() {
        window.addEventListener("message", function(event) {
            if (event.source != window) {
                return;
            }

            if (event.data && (event.data === "travic-sign-start")) {
                logDebug.call(this, "Activation message received.");

                searchAndConvertEmbedTagToHtml5.call(this);
                searchAndConvertObjectTagToHtml5.call(this);
            }
        });
    } 
    
    function startObserver() {
        logDebug("Starting observer to wait for plugin instances to appear");
        
        // Configuration of the observer:
        var observerConfig = {
            childList : true,
            attributes : false,
            characterData : false,
            subtree : true
        };

        // Create an observer instance
        var pluginTagObserver = new MutationObserver(function(mutations) {
            searchAndConvertEmbedTagToHtml5.call(this);
            searchAndConvertObjectTagToHtml5.call(this);
        });

        // Pass in the target node, as well as the observer options
        pluginTagObserver.observe(document, observerConfig);

        // Stop observer on unload
        $(window).on('beforeunload', function(){
            logDebug("Stopping observer for plugin instances");
            pluginTagObserver.disconnect();
        });

        logDebug("Observer started");
    }

    function logDebug(message) {
        if (optionsService.isDebugEnabled()) {
            var dateString = formatDateString(new Date());
            console.debug(dateString + " - " + PRODUCT_NAME + " [ApiConverter]: " + message);
        }
    }

    function logError(message) {
        var dateString = formatDateString(new Date());
        console.error(dateString + " - " + PRODUCT_NAME + " [ApiConverter]: " + message);
    }
    
    return {
            
            init : function() {
                if (isFirefox() && !isFirefoxVersionSufficient()) {
                    logDebug("Firefox version insufficient. At least version " + getMinRequiredFirefoxVersion() + " is required.");
                    return;
                }
                
                logDebug("Initializing API converter");
                
                // Search through the currently loaded page
                searchAndConvertEmbedTagToHtml5();
                searchAndConvertObjectTagToHtml5();

                // Registers an event listener that starts the conversion process on user request
                addGlobalEventListener();

                // Wait for new <embed> and <object>s added to the DOM.
                startObserver();
            }
    };
    
})();
apiConverter.init();
